/*****************************************************************************
 * prodos/pmap.c
 * Apple partition map implementation.
 *
 * Apple II ProDOS Filesystem Driver for Linux 2.4.x
 * Copyright (c) 2001 Matt Jensen.
 * This program is free software distributed under the terms of the GPL.
 *
 * 18-May-2001: Created
 *****************************************************************************/

/*= Kernel Includes =========================================================*/

#include <linux/blkdev.h>

/*= ProDOS Includes =========================================================*/

#include "prodos.h"

/*= Private Functions =======================================================*/

/*****************************************************************************
 * prodos_check_footprint()
 * Determine whether a partition contains a ProDOS filesystem.
 *****************************************************************************/
static int prodos_check_footprint(struct super_block *sb,int start_block) {
	struct buffer_head *bh = NULL;
	struct prodos_dir_block_first *pdbf = NULL;
	int is_prodos = 0;

	/* Load volume directory block. */
	bh = prodos_bread(sb,start_block + PRODOS_VOLUME_DIR_BLOCK);
	if (!bh) {
		PRODOS_ERROR_0(sb,"prodos_bread() failed");
		goto out_error;
	}
	pdbf = (struct prodos_dir_block_first*)bh->b_data;

	/* Check for ProDOS footprint in the block. */
	if (pdbf->header.prev == 0 && PRODOS_GET_STYPE(pdbf->u.vol_header.name) == PRODOS_STYPE_VOLHEADER)
		is_prodos = 1;
	else is_prodos = 0;

	/* Release the block. */
	prodos_brelse(bh);
	bh = NULL;

	/* Return appropriate result code. */
	return is_prodos;

out_error:
	if (bh) {
		prodos_brelse(bh);
		bh = NULL;
	}
	return 0;
}

/*****************************************************************************
 * prodos_find_partition_v1()
 * Find a partition in a V1 Apple partition map.
 *****************************************************************************/
static int prodos_find_partition_v1(struct super_block *sb,struct buffer_head *bh,int part_num) {
	PRODOS_ERROR_0(sb,"old-style partition maps not supported");
	return 0;
}

/*****************************************************************************
 * prodos_find_partition_v2()
 * Find a partition in a V2 Apple partition map.
 *****************************************************************************/
static int prodos_find_partition_v2(struct super_block *sb,struct buffer_head *bh,int part_num) {
	struct prodos_partition_map_v2 *pm = NULL;
	int num_blocks = 0;
	int i = 0;
	int found = 0;
	int prodos_parts_found = 0;

	/* Get total number of entries (blocks) in the partition map. */
	pm = (struct prodos_partition_map_v2*)bh->b_data;
	num_blocks = be32_to_cpu(pm->num_blocks);

	/* V2 partition map holds one entry per block. */
	for (i = 0;!found && i < num_blocks;i++) {
		/* Load next block in partition map (except for first iteration.) */
		if (!bh) {
			bh = prodos_bread(sb,PRODOS_PART_MAP_BLOCK + i);
			if (!bh) {
				PRODOS_ERROR_0(sb,"prodos_bread() failed\n");
				goto out_error;
			}
			pm = (struct prodos_partition_map_v2*)bh->b_data;
		}
	
		/* Check partition for ProDOS footprint. */
		if (!strcmp(PRODOS_PART_TYPE_PRODOS,pm->type) &&
			prodos_check_footprint(sb,be32_to_cpu(pm->start))) {
			/* Check for name or index match. */
			if ((part_num == prodos_parts_found) ||
				(part_num < 0 && !strcmp(PRODOS_SB(sb)->s_part,pm->name))) {
				PRODOS_SB(sb)->s_part_start = be32_to_cpu(pm->start);
				PRODOS_SB(sb)->s_part_size = be32_to_cpu(pm->size);
				found = ~0;
			}
			
			/* Found another ProDOS partition. */
			prodos_parts_found++;
		}

		/* Release this block. */
		prodos_brelse(bh);
		bh = NULL;
		pm = NULL;
	}

	/* Return appropriate result code. */
	return found;
	
out_error:
	if (bh) {
		prodos_brelse(bh);
		bh = NULL;
	}
	return 0;
}

/*= Exported Functions ======================================================*/

/*****************************************************************************
 * prodos_find_partition()
 * Locate a partition in an Apple partition map. Before call, sb->generic_sbp
 * must be set to a pointer to a struct prodos_sb_info; the name (or zero-
 * relative index number in string form) of the partition to find must be in
 * its s_part member. On return, the partition's start position and size are
 * stored in the s_start and s_size fields.
 *****************************************************************************/
int prodos_find_partition(struct super_block *sb) {
	SHORTCUT struct prodos_sb_info * const psb = PRODOS_SB(sb);
	int result = 0;
	struct prodos_2img_header *p2mh = NULL;
	int part_num = -1;
	char *endp = NULL;
	struct buffer_head *bh = NULL;

	/* Still check for image file bounds if s_part is not set. */
	if (!psb->s_part[0]) {
		/* Read block zero to check for image signature. */
		bh = prodos_bread(sb,0);
		if (!bh) {
			PRODOS_ERROR_0(sb,"prodos_bread() failed");
			goto out_error;
		}

		/* Check for 2IMG magic number. */
		p2mh = (struct prodos_2img_header*)bh->b_data;
		if (!memcmp(p2mh->magic,PRODOS_IMAGE_2IMG_MAGIC,strlen(PRODOS_IMAGE_2IMG_MAGIC))) {
			/* Image data must begin on a 512-byte boundary. */
			if (le32_to_cpu(p2mh->data_offset) % PRODOS_BLOCK_SIZE != 0) {
				PRODOS_ERROR_0(sb,"2IMG images must be block-aligned (use mkba2img)");
				goto out_error;
			}

			/* Pretend the data block is a partition. */
			psb->s_part_start = le32_to_cpu(p2mh->data_offset) / 512;
			psb->s_part_size = le32_to_cpu(p2mh->data_len) / 512;
			goto out_success;
		}

		/* Release block zero. */
		prodos_brelse(bh);
		bh = NULL;
	}

	/* s_part may be the textual representation of a number. */
	part_num = simple_strtoul(psb->s_part,&endp,10);
	if ((void*)endp == (void*)psb->s_part) part_num = -1;

	/* Read first block to check for partition map or image header. */
	bh = prodos_bread(sb,PRODOS_HFS_DRIVER_BLOCK);
	if (!bh) {
		PRODOS_ERROR_0(sb,"prodos_bread() failed");
		goto out_error;
	}

	/* Check for a Macintosh driver partition. */
	if (memcmp(bh->b_data,PRODOS_HFS_DRIVER_MAGIC,strlen(PRODOS_HFS_DRIVER_MAGIC))) goto out_error;
	prodos_brelse(bh);

	/* Read first block of the partition map to determine map version. */
	bh = prodos_bread(sb,PRODOS_PART_MAP_BLOCK);
	if (!bh) {
		PRODOS_ERROR_0(sb,"prodos_bread() failed");
		goto out_error;
	}

	/* Dispatch appropriate partition map parser (which will free bh.) */
	if (!memcmp(bh->b_data,PRODOS_PART_MAP_MAGIC_V1,2))
		return prodos_find_partition_v1(sb,bh,part_num);
	else if (!memcmp(bh->b_data,PRODOS_PART_MAP_MAGIC_V2,2))
		return prodos_find_partition_v2(sb,bh,part_num);
	
	/* Partition map format was not recognized. */
	PRODOS_ERROR_0(sb,"unknown partition map format");
	goto out_error;

out_error:
	result = 0;
	goto out_cleanup;
out_success:
	result = ~0;
	goto out_cleanup;
out_cleanup:
	if (bh) {
		prodos_brelse(bh);
		bh = NULL;
	}
	return result;
}
